/*======================================================================*/
/*		TITLE:			TGA2CRY.C			*/
/*		Function:		Convert TGA color data to CRY	*/
/*					data represented in 68000	*/
/*					data files.			*/
/*									*/
/*		First Edit:		04/22/93			*/
/*		Programmer:		Andrew J. Burgess		*/
/*									*/
/*		Additional Help:	Eric Ginner, Mike Pooler	*/
/*									*/
/*		   COPYRIGHT 1993 ATARI CORPORATION.			*/
/*	  UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,		*/
/*	  PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR		*/
/*	THE ASSOCIATED AUDIOVISUAL WORK	IS STRICTLY PROHIBITED.		*/
/*	       		ALL RIGHTS RESERVED.				*/
/*									*/
/*----------------------------------------------------------------------*/
/* This program converts 24 bit TGA picture files to 16 bit CRY or RGB	*/
/* format the TGA file has an 18 byte header, sometimes followed by the	*/
/* filename followed by the RGB data. The header contains the follow-	*/
/* ing data:								*/
/*									*/
/*	Bytes 00-01	how many characters in the filename at the	*/
/*				end of the header			*/
/*	Bytes 02-11	unknown and unused for our purposes		*/
/*	Bytes 12-13	image width (byte swapped)			*/
/*	Bytes 14-15	image height (byte swapped)			*/
/*	Bytes 16	24 if 24 bit data				*/
/*	Bytes 17	???						*/
/*	Bytes 18-??	optional filename based on byte 00-01		*/
/*									*/
/* The RGB data is one byte for red, one byte for green, one byte for	*/
/* blue.								*/
/*									*/
/*	The BOTTOM LINE OF THE IMAGE DATA IS FIRST,			*/
/*		IN THE ORDER BGRBGR...etc				*/
/*									*/
/* but each line has the pixels in left to right order			*/
/*									*/
/*======================================================================*/
/*	main()			- main driver.				*/
/*	Version()		- output version information.		*/
/*	Help()			- output application information.	*/
/*	Initialize()		- global initialization.		*/
/*	Convert()		- convert the TGA to a CRY 68000	*/
/*					assembly file			*/
/*	ReadTGAHeader()		- read in the 18 byte TGA header.	*/
/*	ProcessData()		- convert the  RGB into a CRY image.	*/
/*	Header()		- write out the Header			*/
/*	Footer()		- write out the footer			*/
/*	OpenFiles()		- open the input and output files	*/
/*	CloseFiles()		- close the input and output files	*/
/*	SwapBytes()		- swap the bytes of a U16 value.	*/
/*======================================================================*/

#include <stdio.h>
#include "defs.h"
#include "tga.h"

/*======================================================================*/
/*	EXTERNALS							*/
/*======================================================================*/
extern	STRING		MessageBuffer[];

/*======================================================================*/
/*	GLOBALS								*/
/*======================================================================*/

FILE		*IFile;
FILE		*OFile;
STRING		FileName[ NAMELEN ];
STRING		LabelName[ NAMELEN ];
S16		LabelNumber;
STRING		InFileName[ NAMELEN ];
STRING		OutFileName[ NAMELEN ];

BOOLEAN		FormatOption;
BOOLEAN		LabelOption;
BOOLEAN		LabelNumberOption;
BOOLEAN		OutputOption;
BOOLEAN		SizeOption;
BOOLEAN		VerboseOption;

TGAHDR		header;			/* TGA file header		*/

U16		dataSize;		/* num of bytes after header	*/
STRING		dataName[ NAMELEN ];	/* data file name		*/

U8		buffer[3000];		/* TGA data buffer		*/

U8		*Image;			/* pointer to image buffer	*/

int	atoi();

/*======================================================================*/
/*	main()			- main driver.				*/
/*======================================================================*/

main(argc,argv)
	int	argc;
	char	**argv;
{

	fprintf( stdout, "\n" );
	Version( stdout, FALSE );

	if ( argc < 2 )
		Help();

	Initialize( argc, argv );	/* initialize global data	*/

	Convert();

	funcEnd();

	fprintf( stdout, "\n\tCompleted\n\n" );
	fflush( stdout );

	return( SUCCESS );
}


/*======================================================================*/
/*	Version()		- output version information.		*/
/*======================================================================*/

Version( fileptr, inHeader )
	FILE	*fileptr;
	BOOLEAN inHeader;
{
	fprintf( fileptr, "\tTGA2JAG\tv0.1\t%s\t%s\n", __DATE__, __TIME__ );
	if ( !inHeader )
		fprintf( fileptr, "\t\t\tby Eric D Ginner & Andrew J Burgess\n\n" );
	fflush( fileptr );
}

/*======================================================================*/
/*	Help()			- output application information.	*/
/*======================================================================*/

Help()
{
	fprintf( stdout, "\nUseage:	TGA2CRY [options] filename\n\n" );
	fprintf( stdout, "      This program converts a 24-bit .TGA file to either a .CRY or\n");
	fprintf( stdout, "      .RGB file.\n");
	fprintf( stdout, "      The .CRY file contains CRY data.\n\n");
	fprintf( stdout, "      The .RGB file contains RGB data.\n\n");
	fprintf( stdout, "      filename:\n" );
	fprintf( stdout, "          a .TGA file to be converted\n\n" );
	fprintf( stdout, "      output:\n" );
	fprintf( stdout, "          either a Jaguar .CRY or .RGB bitmap file containing\n" );
	fprintf( stdout, "          680x0 assembly data allocation statements\n\n" );
	fprintf( stdout, "      options:\n" );
	/*               ".........1.........2.........3.........4.........5.........6.........7.........8	*/
	fprintf( stdout, "          -f <format>     output file format, specifies the final image\n" );
	fprintf( stdout, "                          format. where <format> is either CRY or RGB.\n" );
	fprintf( stdout, "                          default: CRY.\n\n" );
	fprintf( stdout, "          -l label        label, specifies the label for the bitmap.\n" );
	fprintf( stdout, "                          the label is an identifier string.\n" );
	fprintf( stdout, "                          an optional number tag can be added\n" );
	fprintf( stdout, "                          using the \"-n\" option below\n" );
	fprintf( stdout, "                          default: <filename> (no suffix)\n\n" );
	fprintf( stdout, "          -n num          label number, specifies the starting label number\n" );
	fprintf( stdout, "                          for use with the label option\n" );
    	fprintf( stdout, "                          the resultant label will be of the form:\n" );
    	fprintf( stdout, "                              <label><num>\n" );
    	fprintf( stdout, "                          where <num> is a 3 digit value (i.e. 000,001)\n" );
	fprintf( stdout, "                          that is incremented everytime another object is\n" );
	fprintf( stdout, "                          encountered\n\n" );
    	fprintf( stdout, "          -o filename     output, specifies the output filename.\n" );
    	fprintf( stdout, "                          this option will create the .CRY file with the\n" );
    	fprintf( stdout, "                          prefix \"filename\".\n" );
    	fprintf( stdout, "                          default: input filename prefix\n\n" );
    	fprintf( stdout, "          -s              size, preceeds the image data with the width and\n" );
    	fprintf( stdout, "                          height of the image in the 1st 2 words.\n" );
    	fprintf( stdout, "                          default: no size.\n\n" );
    	fprintf( stdout, "          -v              verbose, output additional information during\n" );
    	fprintf( stdout, "                          processing. (debugging)\n\n" );
    	fflush( stdout ); 
	exit( WARN );

    	/*               ".........1.........2.........3.........4.........5.........6.........7.........8	*/
};

/*======================================================================*/
/*	Initialize()		- global initialization.		*/
/*======================================================================*/

Initialize(argc,argv)
	int	argc;
	char	**argv;
{
	U16	idx;
	U16	len;
	STRING	*fnptr;
	STRING	*argptr;

	funcInit();

	IFile = NULL;
	OFile = NULL;

	for( idx=0; idx<NAMELEN; ++idx )
	{
		FileName[idx]    = '\0';
		InFileName[idx]  = '\0';
		OutFileName[idx] = '\0';
		LabelName[idx]   = '\0';
	}

	FormatOption      = CRY_FMT;		/* default use internal label		*/
	LabelOption       = FALSE;		/* default use internal label		*/
	LabelNumberOption = FALSE;		/* default no label numbering		*/
	LabelNumber       = 0;			/* default label number			*/
	OutputOption      = FALSE;		/* default use input filename		*/
	SizeOption        = FALSE;		/* default no size			*/
	VerboseOption     = FALSE;		/* default off				*/

	/* process the arguments */
	idx = 1;			/* skip the application name */
	while( idx < (argc-1) )
	{
		argptr = argv[ idx ];	/* get next arguement	*/

		if ( *argptr == '-' )	/* is it an option */
		{
			switch( *(argptr+1) )
			{
			case 'f':	/* label	*/
				++idx;			/* skip "-f" */
				argptr = argv[ idx ];	/* get next arguement	*/

				if ( strcmp( "RGB", argptr ) == 0 )
					FormatOption = RGB_FMT;
				else
				if ( strcmp( "CRY", argptr ) == 0 )
					FormatOption = CRY_FMT;
				else
				if ( strcmp( "rgb", argptr ) == 0 )
					FormatOption = RGB_FMT;
				else
				if ( strcmp( "cry", argptr ) == 0 )
					FormatOption = CRY_FMT;
				else
				{
					sprintf( MessageBuffer, "illegal output file format (%s)", argptr );
					ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
				}

#if (DEBUG)
				fprintf( stdout,"\n\tOption FORMAT: (%s)\n",argptr );
				fflush( stdout );
#endif
				break;
			case 'l':	/* label	*/
				LabelOption = TRUE;
				++idx;			/* skip "-l" */
				argptr = argv[ idx ];	/* get next arguement	*/

#if (DEBUG)
				fprintf( stdout,"\n\tOption LABEL: (%s)\n",argptr );
				fflush( stdout );
#endif
				strcpy( LabelName,argptr );
				break;
			case 'n':	/* label number	*/
				++idx;			/* skip "-n" */
				argptr = argv[ idx ];	/* get next arguement	*/

#if (DEBUG)
				fprintf( stdout,"\n\tOption LABEL NUMBER: (%s)\n",argptr );
				fflush( stdout );
#endif
				LabelNumber = atoi( argptr );
				if ( LabelNumber >= 0 )
					LabelNumberOption = TRUE;
				else
				{
					sprintf( MessageBuffer, "illegal label number option value (%s)", argptr );
					ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
				}
				break;
			case 'o':	/* output	*/
				OutputOption = TRUE;
				++idx;			/* skip "-o" */
				argptr = argv[ idx ];	/* get next arguement	*/

#if (DEBUG)
				fprintf( stdout,"\n\tOption OUTPUT: (%s)\n",argptr );
				fflush( stdout );
#endif
				strcpy( OutFileName,argptr );

                                /* check for a SUFFIX */
				fnptr = OutFileName;

                                while ( *fnptr && (*fnptr != '.') )
                                        ++fnptr;
                                if ( *fnptr != '.' )
                                {
                                        /* add suffix */
					if ( FormatOption ==  RGB_FMT )
	                                        strcat( OutFileName, ".RGB" );
					else
	                                        strcat( OutFileName, ".CRY" );
                                }
				break;
			case 's':	/* size	*/
				SizeOption = TRUE;
#if (DEBUG)
				fprintf( stdout,"\n\tOption SIZE:\n" );
				fflush( stdout );
#endif
				break;
			case 'v':	/* label	*/
				VerboseOption = TRUE;
#if (DEBUG)
				fprintf( stdout,"\n\tOption VERBOSE:\n" );
				fflush( stdout );
#endif
				break;
			default:
				sprintf( MessageBuffer, "unknown option (%s)", argptr );
				ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
			}
                }
                else
                {
			fprintf( stdout, "\nERROR: non-option request (%s)\n", argptr );
			fprintf( stdout, "       input file is final argument\n" );
			fflush( stdout );
			exit( ERROR );
		}
		++idx;
        }

	argptr = argv[ argc-1];		/* last arguement is input file	*/

        /* is either the input filename or an error */
        strcpy( InFileName, argv[ idx ] );

        if ( !OutputOption )
 	{
		strcpy( OutFileName, InFileName );
		fnptr = OutFileName;

		/* check for a SUFFIX and convert to .CRY/.RGB */
		while ( *fnptr && (*fnptr != '.') )
			++fnptr;
		if ( *fnptr == '.' )
		{
			/* NULL '.' and suffix */
			*fnptr = '\0';
		}
	}

	if ( !LabelOption )
 	{

#if (DEBUG)
		if ( VerboseOption )
		{
			sprintf( MessageBuffer, "Converting %s to a label", InFileName );
			DebugMessage( MessageBuffer, __FILE__, __LINE__ );
		}
#endif

		/* find end of name */
		len = strlen( InFileName );
		fnptr = InFileName + (len-1);

		/* search backwards to see if there is a pathname */
		while ( (*fnptr != '\\') && (*fnptr != '/') && (--len)  )
			--fnptr;
		strcpy( LabelName, fnptr );
		fnptr = LabelName;


#if (DEBUG)
		if ( VerboseOption )
		{
			sprintf( MessageBuffer, "Pre-Label: %s", LabelName );
			DebugMessage( MessageBuffer, __FILE__, __LINE__ );
		}
#endif

		/* check for a SUFFIX and strip */
		while ( *fnptr && (*fnptr != '.') )
			++fnptr;
		if ( *fnptr == '.' )
		{
			/* NULL '.' and suffix */
			*fnptr = '\0';
		}
	}
}

/*======================================================================*/
/*	Convert()		- convert the TGA to a CRY 68000	*/
/*					assembly file			*/
/*======================================================================*/

Convert( )
{

#if (FUNCTRACE)
	funcEnter( "Convert()", __FILE__, __LINE__ );
#endif

	/* open the .TGA and .CRY/.RGB files			*/
	OpenFiles();

	/* read the 18 byte header				*/
	ReadTGAHeader();

	/* output the header					*/
	Header();

	/* read in the bitmap data and convert to CRY values	*/
	/* output the values into the .CRY file			*/
	ProcessData();

	/* output the footer					*/
	Footer();

	/* close the .CRY/.RGB and .TGA files			*/
	CloseFiles();

#if (FUNCTRACE)
	funcExit(  "Convert()", __FILE__,__LINE__ );
#endif

}

/*======================================================================*/
/*	ReadTGAHeader()		- read in the 18 byte TGA header.	*/
/*======================================================================*/

ReadTGAHeader()
{

#if (FUNCTRACE)
	funcEnter( "ReadTGAHeader()", __FILE__, __LINE__ );
#endif

	/* read in the TGA header info */
	if ( fread(&header,1,sizeof(TGAHDR),IFile) == 0 )
	{
		if ( ferror(IFile) )
		{
			sprintf( MessageBuffer, "header read error in file (%s)", InFileName );
			ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
		}
  		else
		{
			sprintf( MessageBuffer, "EOF encountered while accessing header in file (%s)\n", InFileName );
			ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
		}
	}

	/* the data in the header is byte swapped */
	SwapBytes( (BYTESWAP*)(&(header.nameSize)) );
	SwapBytes( (BYTESWAP*)(&(header.imageWidth)) );
	SwapBytes( (BYTESWAP*)(&(header.imageHeight)) );

	if( header.bitsPerPixel != DATA24 )
	{
		sprintf( MessageBuffer, "file (%s) is not a 24-bit .TGA file", InFileName );
		ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
	}

#if (FUNCTRACE)
	funcExit(  "ReadTGAHeader()", __FILE__,__LINE__ );
#endif
}

/*======================================================================*/
/*	ProcessData()		- convert the  RGB into a CRY image.	*/
/*======================================================================*/

ProcessData()
{
	U8	red;
	U8	green;
	U8	blue;
	U8	intensity;
	U8	color;
	U16	column;
	U16	completed;
	U16	itemsPerLine;
	U16	rgb;
	S32	line;
	U32	startOfLine;
	float	r,g,b;


#if (FUNCTRACE)
	funcEnter( "ProcessData()", __FILE__, __LINE__ );
#endif

	itemsPerLine = 0;	/* number of values per line in CRY file	*/

	for(line = (header.imageHeight-1); line >= 0; line--)
	{
		/* remember that each pixel in the TGA file is a 3-byte RGB value	*/
/*
		startOfLine = header.nameSize + sizeof(TGAHDR) + (line * (U32)(header.imageWidth * 3));
*/
		startOfLine = header.nameSize + 18 + (line * (U32)(header.imageWidth * 3));

		/* locate the pixel to be processed	*/
		fseek( IFile, startOfLine, SEEK_ABS);	/* SEEK_SET is start of file */
		for(column = 0; column < (header.imageWidth * 3); column += 3)
		{
			/* NOTE: the TGA file has the RGB values in reverse order */

			blue  = fgetc( IFile );
			green = fgetc( IFile );
			red   = fgetc( IFile );

#if (DEBUG)
			if ( VerboseOption )
			{
				sprintf( MessageBuffer, "\t\tRed %3d Green %3d Blue %3d", red, green, blue );
				DebugMessage( MessageBuffer, __FILE__, __LINE__ );
			}
#endif

			if ( FormatOption == RGB_FMT )
			{
				rgb  = (int)(red >> 3) << 5;	/* reduce to 5 bits */
				rgb += (int)blue >> 3;		/* reduce to 5 bits */
				rgb  = rgb << 6;		/* make room for green */
				rgb += (int)green >> 2;

				if( itemsPerLine == 0 )
				{
					/* start a new line */
 						fprintf( OFile, "\tDC.W\t" );
					fprintf( OFile, "$%04x", rgb );
				}
				else
				{
					fprintf( OFile, ",$%04x", rgb );
				}
			}
			else		/* DEFAULT: CRY FORMAT	*/
			{
			  	/* convert RGB to 16 bit CRY value */
				r = (float)red / 255.0;
				g = (float)green / 255.0;
				b = (float)blue / 255.0;
				RGB2CRY_float( r, g, b, &intensity, &color);

#if (DEBUG)
				if ( VerboseOption )
				{
					sprintf( MessageBuffer, "\t\t\t\tColor %04x Intensity %04x", color, intensity );
					DebugMessage( MessageBuffer, __FILE__, __LINE__ );
				}
#endif

				if( itemsPerLine == 0 )
				{
					/* start a new line */
 					fprintf( OFile, "\tDC.W\t" );
					fprintf( OFile, "$%02x%02x", color, intensity );
				}
				else
				{
					fprintf( OFile, ",$%02x%02x", color, intensity );
				}
			}

			++itemsPerLine;

			/* print 8 words per line */
			if( itemsPerLine == 8)
			{
				fprintf( OFile, "\n" );	/* end the current line */
				itemsPerLine = 0;
			}
			fflush( OFile );
		}

		completed = (header.imageHeight - line) * 100 / header.imageHeight;

		fprintf( stdout, "\t%3d%% completed\r", completed);	/* % of lines completed */
		fflush( stdout );
	}

#if (FUNCTRACE)
	funcExit(  "ProcessData()", __FILE__,__LINE__ );
#endif

}

/*======================================================================*/
/*	Header()		- write out the Header			*/
/*======================================================================*/

Header( )
{
	fprintf( OFile, "\n" );
	fprintf( OFile, ";*======================================================================*\n" );
	fprintf( OFile, ";*\n" );
	fprintf( OFile, ";*	File:		%s\n", OutFileName );
	fprintf( OFile, ";*\n" );
	fprintf( OFile, ";*	Created From:	%s\n", InFileName  );
	fprintf( OFile, ";*\n" );
	fprintf( OFile, ";*" );
	Version( OFile, TRUE );
	fprintf( OFile, ";*\n" );
	fprintf( OFile, ";*======================================================================*\n" );
	fprintf( OFile, "\n" );
	fprintf( OFile, "\t.data\n\n\t.even\n\n" );
	fflush( OFile );
	
	fprintf( stdout, "\n\n\timage size: %d x %d image\n",header.imageWidth,header.imageHeight);
	fflush( stdout );

	if (LabelOption)
	{
		if ( LabelNumberOption )
		{
			fprintf( OFile, "\t.globl\t%s%03d\n\n", LabelName,LabelNumber );
			fprintf( OFile, "%s%03d:", LabelName, LabelNumber );
			++LabelNumber;
		}
		else
		{
			fprintf( OFile, "\t.globl\t%s\n\n", LabelName );
			fprintf( OFile, "%s:", LabelName );
		}
	}
	else
	{
		fprintf( OFile, "\t.globl\t%s\n\n", LabelName );
		fprintf( OFile, "%s:", LabelName );
	}
	fprintf( OFile, "\t;* Size ( %d x %d )\n", header.imageWidth, header.imageHeight );

	if ( SizeOption )
		fprintf( OFile, "\tDC.W\t%d,%d,0,0	;* extra 2 words for phrase alignment KLUDGE\n",
				 header.imageWidth, header.imageHeight );

	fflush( OFile );
}


/*======================================================================*/
/*	Footer()		- write out the footer			*/
/*======================================================================*/

Footer()
{
	fprintf( OFile, "\n" );
	fprintf( OFile, ";*======================================================================*\n" );
	fprintf( OFile, "\n\t.end\n\n" );
	fprintf( OFile, ";*======================================================================*\n" );
	fprintf( OFile, ";*                                    EOF\n" );
	fprintf( OFile, ";*======================================================================*\n" );
	fprintf( OFile, "\n" );
	fflush( OFile );

	fprintf( stdout, "\n" );
	fflush( stdout );

}

/*======================================================================*/
/*	OpenFiles()		- open the input and output files	*/
/*======================================================================*/

OpenFiles()
{
	STRING	*tptr;		/* temporary string pointer		*/
	U16	len;

#if (FUNCTRACE)
	funcEnter( "OpenFiles()", __FILE__, __LINE__ );
#endif

	/* check to see if a suffix exists for this filename */
	tptr = InFileName;
	while (*tptr != '\0')
		++tptr;		/* find end of filename */
 	len = 0;
	while( (*tptr != '.') && (len != 4) )
	{
		++len;
		--tptr;
	}

	if (*tptr != '.' )	/* no suffix */
	{
		strcat( InFileName, ".TGA" );

		if ( !OutputOption )
		{
			if ( FormatOption == RGB_FMT )
				strcat( OutFileName, ".RGB" );
			else
				strcat( OutFileName, ".CRY" );
		}
	}
	else
	{
		/* find '.' and override suffix with the format */
		tptr = OutFileName;

		tptr += (strlen( OutFileName )-1);
 		len = 0;
		while( (*tptr != '.') && (len != 4) )
		{
			++len;
			--tptr;
		}

		if (*tptr == '.' )	/* no suffix */
		{
			*tptr = '\0';
		}
		tptr = OutFileName;
			
		if ( FormatOption == RGB_FMT )
			strcat( tptr, ".RGB" );
		else
			strcat( tptr, ".CRY" );
	}


	fprintf( stdout, "\n\tconverting (%s) to (%s)...\n", InFileName, OutFileName );
	fflush( stdout );

	/* open the .TGA input file	*/

	if ((IFile=fopen(InFileName,"rb")) == NULL)
	{
		sprintf( MessageBuffer, "file (%s) could not be opened", InFileName );
		ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
	}

	/* open the output file	*/

	if ((OFile=fopen(OutFileName,"w")) == NULL)
	{
		sprintf( MessageBuffer, "file (%s) could not be created", OutFileName );
		ErrorMessage( MessageBuffer, __FILE__, __LINE__ );
	}


#if (FUNCTRACE)
	funcExit(  "OpenFiles()", __FILE__,__LINE__ );
#endif
}

/*======================================================================*/
/*	CloseFiles()		- close the input and output files	*/
/*======================================================================*/

CloseFiles()
{
#if (FUNCTRACE)
	funcEnter( "CloseFiles()", __FILE__, __LINE__ );
#endif

	if( IFile != NULL )
	{
		fclose( IFile );
		IFile = NULL;
	}
	if( OFile != NULL )
	{
		fclose( OFile );
		OFile = NULL;
	}

#if (FUNCTRACE)
	funcExit(  "CloseFiles()", __FILE__,__LINE__ );
#endif
}

/*======================================================================*/
/*	SwapBytes()		- swap the bytes of a U16 value.	*/
/*======================================================================*/

SwapBytes( temp16 )
	BYTESWAP	*temp16;
{
	U8		temp8;
	U16		value;

#if (FUNCTRACE)
	funcEnter( "SwapBytes()", __FILE__, __LINE__ );
#endif

	temp8          = temp16->loByte;
	temp16->loByte = temp16->hiByte;
	temp16->hiByte = temp8;
	value          = *(U16*)temp16;

#if (FUNCTRACE)
	funcExit(  "SwapBytes()", __FILE__,__LINE__ );
#endif

}

/*======================================================================*/
/*				EOF					*/
/*======================================================================*/
